/**
* \file: AilAudioSinkImpl.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* Actual implementation of AilAudioSink class (see pImpl idiom)
*
* \component: Baidu CarLife
*
* \author: P. Govindaraju / RBEB/GM / Pradeepa.Govindaraju@in.bosch.com
*          P. Acar / ADIT/SW2 / pacar@de.adit-jv.com
*
* \copyright (c) 2016-2017 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/
#include <chrono>

#include <AudioFactory.h>
#include <AudioBackend.h>
#include <AudioTypes.h>
#include <bdcl/BaiduCoreCallbackDealer.h>
#include <bdcl/AditAudioSink.h>
#include "AilAudioSinkImpl.h"

LOG_IMPORT_CONTEXT(bdcl_audio)

namespace adit { namespace bdcl {

// todo you don't have to pull whole namespaces, especially for std. simplify!
using namespace adit::utility::audio;
using namespace adit::utility;
using namespace std;

#define NUMOFBITSPERBYTE 8

// todo this implementation also has thread safety issues. Fix them without changing I/F

// todo get rid of dead code
// #define BUFFER_MONITORING

// todo check State Machine, there is room for improvement

// todo get rid of this pointers, they are just wrong. Also use initialization list, it's cleaner
// todo initialize all parameters
AilAudioSinkImpl::AilAudioSinkImpl(IAditAudioSinkCallbacks* inCallbacks, CoreCallbackDealer* inCallbackDealer)
: mCallbacks(inCallbacks), mChangedConfigParams(0), mIsSetThreadParam(false), mCallbackDealer(inCallbackDealer)
{
    // todo why not just give Alsa as an input parameter? or it can be a const string?
    std::string backendLibName("Alsa");
    mAilBackend = Factory::Instance()->createBackend(backendLibName, *this);

    _setState(audioStatus::initializing);

    mPlayItemTimeMs = 0;
    bufferTimeCurMs = 0;
    mAudioSinkRecordRunning = false;

}

AilAudioSinkImpl::~AilAudioSinkImpl()
{
    // todo call shutdown
    // todo give back memory if there is any left
    teardown();
}

// todo const string?
void AilAudioSinkImpl::setConfigItem(std::string inKey, std::string inValue)
{
    LOGD_DEBUG((bdcl_audio, "setConfigItem: inKey = %s inValue = %s", inKey.c_str(), inValue.c_str()));
    mChangedConfigParams += mConfig.set(inKey, inValue);
}

// todo is there anything to initialize?
bool AilAudioSinkImpl::initialize() {
    // todo investigate state machine
    if (!mConfig.ResultConfig()) {
        LOG_ERROR((bdcl_audio, "%s: could not initialize audio sink endpoint due to configuration error",
                mConfig.mStreamType.c_str()));

        return false;
    }

    if (mConfig.mStreamType == "AUDIO_STREAM_MEDIA")
    {
        mCallbackDealer->registerMediaAudioCallbacks(mCallbacks);
        mCallbackDealer->registerMediaAudioBackendCallbacks(this);

    } else if (mConfig.mStreamType == "AUDIO_STREAM_NAVI_TTS") {

        mCallbackDealer->registerNaviTtsAudioCallbacks(mCallbacks);
        mCallbackDealer->registerNaviTtsAudioBackendCallbacks(this);

    } else if (mConfig.mStreamType == "AUDIO_STREAM_VR_TTS") {

        mCallbackDealer->registerVrTtsAudioCallbacks(mCallbacks);
        mCallbackDealer->registerVrTtsAudioBackendCallbacks(this);
    }

    return true;
}

// todo is there anything to teardown?
void AilAudioSinkImpl::teardown()
{
    _setState(audioStatus::shuttingDown);
    playbackStop(true);

    if (mAudioSinkRecordRunning) {
        mAudioSinkRecord.close();
        mAudioSinkRecordRunning = false;
    }
}

bool AilAudioSinkImpl::playbackStart()
{
    // todo there is no init but only playbackStart now, buffer cleaning cannot happen here
    // ensure to not play old buffers on new initialization. should be done by playbackStop
//    mAudioItemDeque.clear();
//    bufferSizeCur = 0;

    LOGD_DEBUG((bdcl_audio, "%s: Application commanded to start playback, current state: %u",
            mConfig.mStreamType.c_str(), static_cast<uint32_t>(_getState())));

    // todo check if already started?
    if (!_startStream())
    {
        return false;
    }

    if (_getState() == audioStatus::buffering && (bufferTimeCurMs >= mConfig.mBufferMinMs))
    {
        _streamToAil();
    }
    /* open the WAV file to store the AudioSink record */
    if( mConfig.mAudioRecord ){
        if (0 != mAudioSinkRecord.open(mConfig.mAudioRecordFile , mConfig.mStreamType.c_str())) {
            LOG_ERROR((bdcl_audio, "%s, playbackStartCallback()  Open AudioSinkRecord file failed.",
                    mConfig.mStreamType.c_str()));
        } else {
            mAudioSinkRecordRunning = true;
        }
    }
    return true;
}

bool AilAudioSinkImpl::playbackStop(bool inFlushBuffer)
{
    bool rc = true;
    LOGD_DEBUG((bdcl_audio, "%s: Application commanded to stop playback, current state: %u",
            mConfig.mStreamType.c_str(), static_cast<uint32_t>(_getState())));

    // todo check if started, otherwise just flush buffer maybe
    // todo DLT log levels are not correct, fix them
    if (_getState() == audioStatus::streaming)
    {
        rc = _stopStream(inFlushBuffer);
    } else if (_getState() == audioStatus::buffering) {
        rc = _stopStream(true);
    } else if (inFlushBuffer) {
        LOGD_DEBUG((bdcl_audio, "%s: Playback not started, just flushing buffer", mConfig.mStreamType.c_str()));
        _flushBuffer();
    } else {
        LOGD_DEBUG((bdcl_audio, "%s: Playback not started", mConfig.mStreamType.c_str()));
    }

    /* insert WAV header and close the AudioSink record*/
    if ((mConfig.mAudioRecord) && mAudioSinkRecordRunning) {
        if (0 != mAudioSinkRecord.createWavHeader(1, mConfig.mNumofChannels, mConfig.mSampleFormat, mConfig.mSampleRate)) {
            LOG_ERROR((bdcl_audio, "%s, playbackStopCallback() create wav header for AudioSinkRecord failed.",
                        mConfig.mStreamType.c_str()));
        }
        mAudioSinkRecord.close();
        mAudioSinkRecordRunning = false;
    }
    return rc;
}

// todo didn't clean up this, should be reviewed
void AilAudioSinkImpl::onAudioDataAvailable(uint8_t *inData, uint32_t inLen)
{
    if (!mIsSetThreadParam) {
        _setThreadPriority();
        mIsSetThreadParam = true;   // setting thread priority only once
    }

    std::unique_lock<std::mutex> guard(audioMut);

    unsigned int num_of_discarded_samples = 0;

    // remove buffers from queue until we have enough space for incoming buffer
    if((bufferTimeCurMs + _convByteToTime(inLen, mConfig.mSampleRate, mConfig.mNumofChannels, mConfig.mSampleFormat)) > mConfig.mBufferMaxMs)
    {
        // todo why delete half of the buffer?
        while(bufferTimeCurMs > mConfig.mBufferMaxMs / 2)
        {
            if(!mAudioItemDeque.empty())
            {
                // todo leaking AudioItems, delete the ones you popped from deque
                bufferTimeCurMs -= _convByteToTime(mAudioItemDeque.front()->mLen, mAudioItemDeque.front()->mSampleRate, mAudioItemDeque.front()->mChannelNumber,
                        mAudioItemDeque.front()->mSampleFormat);
                mAudioItemDeque.pop_front();
                num_of_discarded_samples++;
            }
            else
            {
                LOG_ERROR((bdcl_audio, "%s: bufferTimeCurMs reports fill level but mAudioItemDeque is empty",
                        mConfig.mStreamType.c_str()));
                bufferTimeCurMs = 0;
            }
        }

        if(num_of_discarded_samples > 0)
        {
            LOGD_DEBUG((bdcl_audio,
                "%s: Number of samples discarded due to buffer overflow = %u bufferTimeCurMs: %u bufferSizeMax: %u len: %u",
                mConfig.mStreamType.c_str(), num_of_discarded_samples, bufferTimeCurMs, mConfig.mBufferMaxMs, inLen));
        }
    }

    // add new incoming sample to the deque
    uint8_t *internalbuf = new uint8_t[inLen];
    std::shared_ptr<AudioItem> audio_item(new AudioItem(this, internalbuf, inLen, mConfig.mSampleRate, mConfig.mNumofChannels, mConfig.mSampleFormat));
    memcpy(audio_item->mDataPtr, inData, inLen);
    mAudioItemDeque.push_back(audio_item);
    bufferTimeCurMs += _convByteToTime(inLen, mConfig.mSampleRate, mConfig.mNumofChannels, mConfig.mSampleFormat);

    LOGD_VERBOSE((bdcl_audio, "%s: Audio data arrived, buffer size: %u ms", mConfig.mStreamType.c_str(), bufferTimeCurMs));

    guard.unlock();
    cvAudioAvailable.notify_one();

    // start streaming to AIL if buffer size has reached the threshold against underrun
    if (_getState() == audioStatus::buffering && (bufferTimeCurMs >= mConfig.mBufferMinMs))
    {
        _streamToAil();
    }
    /* record received AudioSink data into WAV file
     * in case Application enables the feature */
    if ((mConfig.mAudioRecord) && mAudioSinkRecordRunning) {
        int res = mAudioSinkRecord.write(inData, (unsigned int)inLen);
        if (res != (int)inLen) {
            LOG_ERROR((bdcl_audio, "%s write data for AudioSinkRecord failed=%d",
                    mConfig.mStreamType.c_str(), res));
        }
    }
}

void AilAudioSinkImpl::onPlaybackInterruptBackend()
{
    if(_getState() == audioStatus::draining ){
        LOGD_DEBUG((bdcl_audio, "%s: Interrupting the draining buffer to flush", mConfig.mStreamType.c_str()));
        _setState(audioStatus::interrupting);
        cvAudioStatus.notify_one();
        LOGD_VERBOSE((bdcl_audio, "%s: AIL notified draining buffer to flush", mConfig.mStreamType.c_str()));
    }else if(_getState() == audioStatus::streaming || _getState() == audioStatus::buffering){
        LOGD_DEBUG((bdcl_audio, "%s: stopping the playback by flushing the buffer onPlaybackInterruptCallback", mConfig.mStreamType.c_str()));
        _setState(audioStatus::interrupting);
        _stopStream(true);
    }else{
        LOGD_DEBUG((bdcl_audio, "%s: onPlaybackInterruptBackend no action has taken state: %u",
                mConfig.mStreamType.c_str(), static_cast<uint32_t>(_getState())));
    }
    mCallbacks->onPlaybackInterrupt();
}

void AilAudioSinkImpl::_setThreadPriority()
{
    if(!mConfig.mDisablePrio)
    {
        /* set thread priority */
        int err = 0;
        int schedPolicy;
        struct sched_param schedParam;

        err = pthread_getschedparam(pthread_self(), &schedPolicy, &schedParam);
        if(err == 0)
        {
            schedParam.sched_priority = mConfig.mThreadPrio;
            err = pthread_setschedparam(pthread_self(), SCHED_FIFO, &schedParam);
            if(err != 0)
            {
                LOG_ERROR((bdcl_audio, "%s: Failed to set thread priority with error %d", mConfig.mStreamType.c_str(),
                        err));
            }
        }
        else
        {
            LOG_ERROR((bdcl_audio, "%s: Failed to get thread priority with error %d", mConfig.mStreamType.c_str(),
                    err));
        }
    }
}

// Below here done except for todos

void AilAudioSinkImpl::error(const std::string& inData) const
{
    LOG_ERROR((bdcl_audio, "%s: %s", mConfig.mStreamType.c_str(), inData.c_str()));
}

void AilAudioSinkImpl::warning(const std::string& inData) const
{
    LOG_WARN((bdcl_audio, "%s: %s", mConfig.mStreamType.c_str(), inData.c_str()));
}

void AilAudioSinkImpl::info(const std::string& inData) const
{
    LOG_INFO((bdcl_audio, "%s: %s", mConfig.mStreamType.c_str(), inData.c_str()));
}

void AilAudioSinkImpl::debug(const std::string& inData) const
{
    LOGD_DEBUG((bdcl_audio, "%s: %s", mConfig.mStreamType.c_str(), inData.c_str()));
}

eLogLevel AilAudioSinkImpl::checkLogLevel() const
{
    return eLogLevel::LL_MAX;
}

// todo should be reviewed
// todo why do I need to do mWasPlayed, most probably copy paste from AAUTO again, here it's memcopied
AudioState AilAudioSinkImpl::processing(unsigned char *in, unsigned char **out, uint32_t &frames)
{
    (void)in; // in is not used for playback
    frames = 0;
    std::unique_lock<std::mutex> guard(audioMut);

    // handle if there is a first element marked as played
    if (!mAudioItemDeque.empty())
    {
        std::shared_ptr<AudioItem> work = mAudioItemDeque.front();

        if (work->mWasPlayed)
        {
            bufferTimeCurMs -= _convByteToTime(work->mLen, work->mSampleRate, work->mChannelNumber, work->mSampleFormat);
            delete[] work->mDataPtr;
            mAudioItemDeque.pop_front();
            if (mAudioItemDeque.empty())
            {
                LOGD_DEBUG((bdcl_audio, "%s: Last element of audio buffer is played", mConfig.mStreamType.c_str()));
                // todo talk to multimedia about the 8 ms extra, avoid the magic number
                cvAudioAvailable.wait_for(guard, std::chrono::milliseconds(mPlayItemTimeMs + 8));
            } else {
                LOGD_VERBOSE((bdcl_audio, "%s: Audio item deleted from buffer, buffer size: %u ms",
                        mConfig.mStreamType.c_str(), bufferTimeCurMs));
            }
        }
    }
    else
    {
        LOGD_DEBUG((bdcl_audio, "%s: No data to play in the buffer, buffer size: %u ms", mConfig.mStreamType.c_str(),
                bufferTimeCurMs));
        bufferTimeCurMs = 0;
        // todo the waiting duration does not make any sense, fix it! It is again copied and pasted from Android Auto
        // The original idea behind it is having the time for one Gal frame + one period (8ms)
        // MUM domain kind of suggested to have periodMs * 3 (24ms) if I didn't misunderstand, but this would be much
        // less than what we have here (and for AAUTO: AAUTO has for 16kHz 64+8 and for 44kHz ~43+8)
        cvAudioAvailable.wait_for(guard, std::chrono::milliseconds(mPlayItemTimeMs + 8));
    }
    // play the first element in the queue
    if (!mAudioItemDeque.empty())
    {
        std::shared_ptr<AudioItem> work = mAudioItemDeque.front();
        *out = work->mDataPtr;
        frames = work->mLen / (mConfig.mNumofChannels * ((mConfig.mSampleFormat)/NUMOFBITSPERBYTE));                                               //((mConfig.mNumBits / 8) * mConfig.mNumChannels);
        work->mWasPlayed = true;
    }

#ifdef BUFFER_MONITORING
    float buffer_sec = (float)bufferSizeCur / 48000.0 / 2.0;
    LOG_ERROR((bdcl_audio,"fill level: %.3f",buffer_sec));
#endif

    // todo careful about the relation between bufferSizeCur and deque.empty
    // draining audio data in the buffer
    if (_getState() == audioStatus::draining)
    {
        LOGD_DEBUG((bdcl_audio, "%s: !playbackStarted and buffer size: %u", mConfig.mStreamType.c_str(),
                bufferTimeCurMs));
        if(bufferTimeCurMs == 0)
        {
            LOGD_DEBUG((bdcl_audio, "%s: Played last sample in stream, stopping streaming",
                    mConfig.mStreamType.c_str()));
            _setState(audioStatus::buffering);
            cvAudioStatus.notify_one();
            LOGD_VERBOSE((bdcl_audio, "%s: AIL notifies drained buffer", mConfig.mStreamType.c_str()));

            // todo return AudioState::STOP; actually
            return AudioState::CONTINUE;
        }
    }
//    } else {
    return AudioState::CONTINUE;
//    }

}

void AilAudioSinkImpl::statistics(const adit::utility::audio::StreamStatistics &status)
{
    (void)status;
    // todo [easy, later] implement body using actual StreamStatistics structure rather than printing its address
}

void AilAudioSinkImpl::eostreaming(const AudioError error)
{
    LOGD_VERBOSE((bdcl_audio, "%s: AIL notifies EOS", mConfig.mStreamType.c_str()));
    if (error != AudioError::OK)
    {
        LOG_ERROR((bdcl_audio, "%s: eostreaming(): Streaming has stopped unexpectedly: %u", mConfig.mStreamType.c_str(),
                (uint32_t)error));
        // todo [easy, later] why not invoke an onError callback to MC application?
        // todo initiate shuttingDown from here
    }
}

bool AilAudioSinkImpl::_startStream()
{
    if (mChangedConfigParams > 0)
    {
        if (_getState() == audioStatus::streaming)
        {
            LOGD_DEBUG((bdcl_audio, "%s: Configuration has changed, restarting audio stream",
                    mConfig.mStreamType.c_str()));
            _stopStream(true);

        } else if (_getState() == audioStatus::buffering) {
            if (int err = mAilBackend->closeStream() != AudioError::OK)
            {
                LOG_ERROR((bdcl_audio, "%s, closeStream() failed in buffering state, error: %u",
                        mConfig.mStreamType.c_str(), static_cast<uint32_t>(err)));
            } else {
                LOGD_VERBOSE((bdcl_audio, "%s, AIL closeStream() succeeded in buffering state",
                        mConfig.mStreamType.c_str()));
            }
            _flushBuffer();
        }

        if (!mConfig.ResultConfig()) {
            LOG_ERROR((bdcl_audio, "%s: could not initialize stream due to configuration error",
                    mConfig.mStreamType.c_str()));

            return false;
        }
        mChangedConfigParams = 0;
    } else if (_getState() == audioStatus::buffering || _getState() == audioStatus::streaming) {
        LOGD_DEBUG((bdcl_audio, "%s: Mobile device sent messages in an unexpected order, dropping the latter",
                mConfig.mStreamType.c_str()));
        return true;
    }

    /* todo the waiting duration does not make any sense, fix it! It is again copied and pasted from Android Auto
     * The original idea behind it is having the time for one GalFrame + one period (8ms)
     * 2048 is defined by GalReceiver as samples per GalFrame
     * MUM domain kind of suggested to have periodMs * 3 (24ms) if I didn't misunderstand, but this would be much
     * less than what we have here (and for AAUTO: AAUTO has for 16kHz 64+8 and for 44kHz ~43+8) */
    mPlayItemTimeMs = 2048 / (mConfig.mSampleRate / 1000);

    // number of samples on one period of jitter buffer
    uint32_t period_samples = mConfig.mPeriodMs * (mConfig.mSampleRate / 1000);
    std::string invalid_capture_device("");

    if(!mConfig.mDisablePrio)
    {
        mAilBackend->setThreadSched(SCHED_FIFO, mConfig.mThreadPrio);
    }

    if(mConfig.mInitToutMs > 0)
    {
        mAilBackend->setInitialTimeout(mConfig.mInitToutMs);
    }

    /* todo Check if Baidu stack has defined any value for ducking audio or whether it is sending ducked audio. This
     * looks like copied code without any thought. */
    mAilBackend->setFadeTime(FadeMode::OUT, StreamDirection::OUT, 0);

    LOGD_DEBUG((bdcl_audio, "Device name to be opened for audio channel %s: %s", mConfig.mStreamType.c_str(),
            mConfig.mDeviceName.c_str()));
    /* todo AudioFormat::S16_LE means 16 bit little endian, which is by luck the same as BDCL although it is again
     * copied from AAUTO. Make it configurable or at least define as macro */
    AudioError err = mAilBackend->openStream(invalid_capture_device, mConfig.mDeviceName, AudioFormat::S16_LE,
            mConfig.mSampleRate, mConfig.mNumofChannels, period_samples);
    if (err != AudioError::OK)
    {
        LOG_ERROR((bdcl_audio, "%s: AIL openStream() failed, error: %u", mConfig.mStreamType.c_str(),
                static_cast<uint32_t>(err)));
        return false;

    } else {
        LOGD_DEBUG((bdcl_audio, "%s: AIL openStream() succeeded", mConfig.mStreamType.c_str()));
    }

    _setState(audioStatus::buffering);

    return true;
}

void AilAudioSinkImpl::_streamToAil()
{
    LOGD_DEBUG((bdcl_audio, "%s: Buffer reached threshold, starting stream: %u ms >= %u ms, current state: %u",
            mConfig.mStreamType.c_str(), bufferTimeCurMs, mConfig.mBufferMinMs, static_cast<uint32_t>(_getState())));

    AudioError err = mAilBackend->startStream();
    if (err != AudioError::OK)
    {
        LOG_ERROR((bdcl_audio, "%s: AIL startStream() failed, error: %u", mConfig.mStreamType.c_str(),
                static_cast<uint32_t>(err)));
    }
    else
    {
        _setState(audioStatus::streaming);
        LOGD_VERBOSE((bdcl_audio, "%s: AIL startStream() succeeded", mConfig.mStreamType.c_str()));
    }
}

bool AilAudioSinkImpl::_stopStream(bool inFlushBuffer)
{
    AudioError err = AudioError::OK;
    bool ret_val = true;

    // todo only abort / stop if alsa device streaming is started
    // abort AIL device if shutting down
    if (_getState() == audioStatus::shuttingDown)
    {
        err = mAilBackend->abortStream();
        if (err != AudioError::OK)
        {
            LOG_ERROR((bdcl_audio, "%s, AIL abortStream() failed, error: %u", mConfig.mStreamType.c_str(),
                    static_cast<uint32_t>(err)));
            ret_val = false;
        } else {
            LOGD_VERBOSE((bdcl_audio, "%s, AIL abortStream() succeeded", mConfig.mStreamType.c_str()));
        }
    // stop AIL device otherwise
    } else {
        // but wait first for the buffer to be drained if it is not to be flushed
        if (!inFlushBuffer)
        {
            std::unique_lock<std::mutex> status_guard(mutAudioStatus);
            mStatus = audioStatus::draining;

            LOGD_VERBOSE((bdcl_audio, "%s: _stopStream:: Before waiting for CV: Buffer is being drained",
                    mConfig.mStreamType.c_str()));
            // todo I couldn't convert the boolean value to a predicate
            while (mStatus == audioStatus::draining) {
                cvAudioStatus.wait(status_guard);
            }
            LOGD_VERBOSE((bdcl_audio, "%s: _stopStream:: CV got notified: Buffer is drained",
                    mConfig.mStreamType.c_str()));
        }

        // todo AIL should actually be stopped in this case, but AIL::stopStream() is buggy, therefore using abort
        err = mAilBackend->abortStream();
        if (err != AudioError::OK)
        {
            LOG_ERROR((bdcl_audio, "%s, AIL stopStream() failed, error: %u", mConfig.mStreamType.c_str(),
                    static_cast<uint32_t>(err)));
            ret_val = false;
        } else {
            LOGD_VERBOSE((bdcl_audio, "%s, AIL stopStream() succeeded", mConfig.mStreamType.c_str()));
        }
        // todo Why setting state to buffering?
        _setState(audioStatus::buffering);
    }

    // finally close AIL device
    // todo only close if alsa device is already opened (mPlaybackStarted most probably)
    err = mAilBackend->closeStream();
    if (err != AudioError::OK)
    {
        LOG_ERROR((bdcl_audio, "%s, closeStream() failed, error: %u", mConfig.mStreamType.c_str(),
                static_cast<uint32_t>(err)));
        ret_val = false;
    } else {
        LOGD_VERBOSE((bdcl_audio, "%s, AIL closeStream() succeeded", mConfig.mStreamType.c_str()));
    }
    _setState(audioStatus::initializing);

    _flushBuffer();
    return ret_val;
}

void AilAudioSinkImpl::_flushBuffer()
{
    // todo leaking allocated AudioItems, pop each and every one, delete[] ptr.
    std::unique_lock<std::mutex> guard(audioMut);
    size_t num_of_discarded = mAudioItemDeque.size();
    mAudioItemDeque.clear();
    bufferTimeCurMs = 0;
    guard.unlock();

    LOGD_DEBUG((bdcl_audio, "%s: Flushed %zu items in audio buffer, current state: %u", mConfig.mStreamType.c_str(),
            num_of_discarded, static_cast<uint32_t>(_getState())));
}

AilAudioSinkImpl::audioStatus AilAudioSinkImpl::_getState()
{
    std::unique_lock<std::mutex> status_guard(mutAudioStatus);
    return mStatus;
}

void AilAudioSinkImpl::_setState(AilAudioSinkImpl::audioStatus inStatus)
{
    std::unique_lock<std::mutex> status_guard(mutAudioStatus);
    LOGD_DEBUG((bdcl_audio, "%s: Transition from state: %u to state: %u", mConfig.mStreamType.c_str(),
            static_cast<uint32_t>(mStatus), static_cast<uint32_t>(inStatus)));
    mStatus = inStatus;
}

unsigned int AilAudioSinkImpl::_convByteToTime(uint32_t inLen, size_t sampleRate, size_t channelNumber, size_t sampleFormat)
{
    uint32_t frames = inLen / (channelNumber * ((sampleFormat)/NUMOFBITSPERBYTE));                 // formula to calculate frames from bytes. frames = (bytes received / ((NumBits / 8) * NumChannels));
    return ( (frames * 1000) / sampleRate);
}

} } /* namespace adit { namespace bdcl */
